// ==============================
// Poller: fetch messages from hq_messages for this station
// - Pending inbox: fetch_hq_messages.php (acknowledged = 0)
// - History (last 3): fetch_hq_log.php
// ==============================
(function () {
  // Map aid station dropdown text to station IDs used in hq_messages.station_target
  function stationNameToId(name) {
    if (!name) return "";
    const trimmed = name.trim().toUpperCase();

    switch (trimmed) {
      case "HELL HILL AID #1":
        return "AS1";
      case "HELL HILL AID #2":
        return "AS2";
      case "HELL HILL AID #3":
        return "AS3";
      case "DANIELSON RANCH":
        return "AS4";
      case "YERBA BUENA":
        return "AS5";
      case "SYCAMORE CANYON":
        return "AS6";
      case "START":
        return "START";
      case "FINISH LINE":
        return "FINISH";
      default:
        return "";
    }
  }

  // Track last seen HQ message ID per station (for pending inbox)
  let lastHqMessageId = 0;

  // Pending inbox polling: only NOT ACKed messages (acknowledged = 0)
  async function pollHqMessagesForStation() {
    const stationId = (window.TVEMC_STATION_ID || "").trim();
    if (!stationId) {
      // No station id set; nothing to poll
      return;
    }

    try {
      const url =
        "fetch_hq_messages.php?station=" +
        encodeURIComponent(stationId) +
        "&since_id=" // +
     //   encodeURIComponent(lastHqMessageId);

      const response = await fetch(url, {
        method: "GET",
        headers: {
          Accept: "application/json"
        }
      });

      if (!response.ok) {
        console.error("fetch_hq_messages HTTP error", response.status);
        return;
      }

      const data = await response.json();
      if (!data.success) {
        console.error("fetch_hq_messages API error", data.error);
        return;
      }

      const messages = data.messages || [];
      if (!messages.length) {
        return;
      }

      messages.forEach((msg) => {
        // Update last seen ID
        if (msg.id && msg.id > lastHqMessageId) {
          lastHqMessageId = msg.id;
        }

        // Call your already-working UI helper
        if (typeof window.showHqMessageAtStation === "function") {
          window.showHqMessageAtStation({
            text: msg.message_text,
            channel: msg.channel,
            operator: msg.operator,
            created_at: msg.created_at,
            id: msg.id
          });
        }
      });
    } catch (err) {
      console.error("fetch_hq_messages fetch error", err);
    }
  }

  // History loader: last 3 messages for this station (including broadcasts)
  async function loadStationHistory() {
    const stationId = (window.TVEMC_STATION_ID || "").trim();
    if (!stationId) return;

    const historyBox = document.getElementById("stationHistory");
    const historyBody = document.getElementById("stationHistoryBody");
    if (!historyBox || !historyBody) return;

    try {
      const params = new URLSearchParams();
      params.set("station", stationId);
      params.set("limit", "10"); // last 3 messages for this station (including ALL)
      const url = "fetch_hq_log.php?" + params.toString();

      const response = await fetch(url, {
        method: "GET",
        headers: { Accept: "application/json" }
      });

      if (!response.ok) {
        console.error("station history HTTP error", response.status);
        return;
      }

      const data = await response.json();
      if (!data.success) {
        console.error("station history API error", data.error);
        return;
      }

      const messages = data.messages || [];

      historyBox.style.display = "block";
      historyBody.innerHTML = "";

      if (!messages.length) {
        const div = document.createElement("div");
        div.style.fontStyle = "italic";
        div.style.color = "#ccc";
        div.textContent = "No recent HQ messages.";
        historyBody.appendChild(div);
        return;
      }

      messages.forEach((msg) => {
        const div = document.createElement("div");
        div.style.borderBottom = "1px solid #444";
        div.style.padding = "2px 0";

        const time = msg.created_at || "";
        const text = msg.message_text || "";
        const channel = (msg.channel || "").toUpperCase();
        const operator = msg.operator || "";

        div.textContent = `[${time}] ${text} (${channel})`;
        if (operator) {
          div.textContent += ` – ${operator}`;
        }

        historyBody.appendChild(div);
      });
    } catch (err) {
      console.error("station history fetch error", err);
    }
  }

  // Start polling when page is loaded (aid station view)
  window.addEventListener("load", function () {
    // Do NOT poll in HQ mode (?hq=1)
    if (window.location.search.includes("hq=1")) {
      return;
    }

    // Read station from the existing Aid Station dropdown
    const stationSelect = document.getElementById("aidStation");
    if (stationSelect && stationSelect.value) {
      const stationName = stationSelect.value; // e.g., "HELL HILL AID #1"
      const stationIdFromName = stationNameToId(stationName);
      console.log(
        "AidStation dropdown: name=",
        stationName,
        "mapped id=",
        stationIdFromName
      );
      if (stationIdFromName) {
        window.TVEMC_STATION_ID = stationIdFromName;
      }
    }

    const stationId = (window.TVEMC_STATION_ID || "").trim();
    if (!stationId) {
      // No station chosen / mapped; do not poll
      return;
    }

    // Optional: load last seen id from localStorage if you want persistence
    const stored = localStorage.getItem(
      "TVEMC_lastHqMessageId_" + stationId
    );
    if (stored) {
      lastHqMessageId = parseInt(stored, 10) || 0;
    }

    // Pending inbox polling (unACKed messages)
    pollHqMessagesForStation();
    setInterval(pollHqMessagesForStation, 30000);

    // Station history (last 3 messages, including ACKed)
    loadStationHistory();
    setInterval(loadStationHistory, 30000);

    // Persist last seen id occasionally
    setInterval(function () {
      if (!stationId) return;
      localStorage.setItem(
        "TVEMC_lastHqMessageId_" + stationId,
        String(lastHqMessageId)
      );
    }, 60000);
  });
})();
