console.log('JavaScript loaded successfully');

let entries = [];
let messageLog = [];
let bibList = [];
let offlineQueue = [];
let editingIndex = null;
let bibSortDirection = Array(15).fill(1);
let messageSortDirection = Array(3).fill(1);
let currentSortColumn = null;
let currentSortDirection = 1;
const distances = ['30K', '50K', '50 mile', '100K', '100 mile', '200 mile', 'DNS'];
const serverURL = '/load_bib.1.3.3.php';
let isSubmitting = false;
let showAlerts = localStorage.getItem('showAlerts') !== 'false';  // Default to true, remember setting

async function isOnline() {
    try {
        // Use ROOT path — load_bib.1.3.3.php is in /var/www/html/
        const response = await fetch('load_bib.1.3.3.php?limit=500',{
            method: 'GET',
            cache: 'no-store'
        });
        return response.ok;
    } catch (error) {
        return false;
    }
}

// Aid Station: show a new message from HQ
window.showHqMessageAtStation = function (msg) {
  // msg can be a string OR an object { text, channel, created_at, operator, id }
  const inbox = document.getElementById("stationInbox");
  const body = document.getElementById("stationInboxBody");
  const title = document.getElementById("stationInboxTitle");
  if (!inbox || !body) return;

  // Ensure box is visible
  inbox.style.display = "block";

  // If this is the first real message, clear the placeholder
  if (body.textContent.trim() === "No messages from HQ yet.") {
    body.textContent = "";
  }

  // Normalize msg
  let text = "";
  let channel = "";
  let created = "";
  let operator = "";
  let messageId = null;

  if (typeof msg === "string") {
    text = msg;
  } else if (msg && typeof msg === "object") {
    text = msg.text || "";
    channel = msg.channel || "";
    created = msg.created_at || "";
    operator = msg.operator || "";
    messageId = msg.id || null;   // <-- DB id from fetch_hq_messages.php
  }

  const ts = created || new Date().toLocaleTimeString();

  // Container for this message
  const wrapper = document.createElement("div");
  wrapper.style.borderBottom = "1px solid #555";
  wrapper.style.padding = "4px 0";
  wrapper.style.display = "flex";
  wrapper.style.alignItems = "center";
  wrapper.style.gap = "8px";

  // Left side: text
  const msgText = document.createElement("div");
  msgText.style.flex = "1";
  msgText.textContent = `[${ts}] ${text}`;
  if (channel) {
    msgText.textContent += ` (via ${channel.toUpperCase()})`;
  }
  if (operator) {
    msgText.textContent += ` – ${operator}`;
  }

  // Right side: "Received" checkbox
  const ackLabel = document.createElement("label");
  ackLabel.style.fontSize = "12px";
  ackLabel.style.whiteSpace = "nowrap";

  const ackCheckbox = document.createElement("input");
  ackCheckbox.type = "checkbox";
  ackCheckbox.style.marginRight = "4px";

  ackLabel.appendChild(ackCheckbox);
  ackLabel.appendChild(document.createTextNode("Received"));

  // When the station checks "Received"
  ackCheckbox.addEventListener("change", function () {
    if (ackCheckbox.checked) {
      // Stop flashing once they acknowledge
      stopHqInboxFlash();

      // Tell the server we've ACK'd this message
      if (messageId) {
        sendHqAckToServer(messageId);
      }
    }
  });

 // wrapper.appendChild(msgText);
 // wrapper.appendChild(ackLabel);
 // body.appendChild(wrapper);

  // Start flashing to draw attention
 // startHqInboxFlash();
//};

  wrapper.appendChild(msgText);
  wrapper.appendChild(ackLabel);

  // Put newest message at the TOP of the inbox
  if (body.firstChild) {
    body.insertBefore(wrapper, body.firstChild);
  } else {
    body.appendChild(wrapper);
  }

  // Start flashing to draw attention
  startHqInboxFlash();
};


// Send ACK back to server so HQ can see message was received
async function sendHqAckToServer(messageId) {
  if (!messageId) return;

  try {
    const response = await fetch("hq_ack_message.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify({ id: messageId })
    });

    if (!response.ok) {
      console.error("hq_ack_message HTTP error", response.status);
      return;
    }

    const data = await response.json();
    if (!data.success) {
      console.error("hq_ack_message API error", data.error);
      return;
    }

    console.log("HQ message ACK stored, id:", messageId);
  } catch (err) {
    console.error("hq_ack_message fetch error", err);
  }
}

// Start flashing the inbox
function startHqInboxFlash() {
  const inbox = document.getElementById("stationInbox");
  if (!inbox) return;
  inbox.classList.add("hq-inbox-new");
}

// Stop flashing (used when they scroll or acknowledge)
function stopHqInboxFlash() {
  const inbox = document.getElementById("stationInbox");
  if (!inbox) return;
  inbox.classList.remove("hq-inbox-new");
}

// Stop flashing when the station scrolls the inbox
window.addEventListener("load", function () {
  const body = document.getElementById("stationInboxBody");
  if (!body) return;

  body.addEventListener("scroll", function () {
    stopHqInboxFlash();
  });
});


async function submitDistanceChange(bib, newDistance, oldDistance) {
    // First: update the actual distance in DB
    const updatePayload = {
        action: 'distance_change',
        bib: parseInt(bib),
        distance: newDistance,
        previous_distance: oldDistance
    };

    // Second: send a fake "IN" entry so it appears in the log with DNS
    const logPayload = {
        bib: parseInt(bib),
        status: `DNS | ${newDistance === 'DNS' ? 'Did Not Start' : 'Distance Change'} | Today | ${document.getElementById('aidStation')?.value || ''} | Distance: ${newDistance} | ${document.getElementById('operatorName')?.value.trim() || 'Unknown'}`,
        distance: newDistance,
        previous_distance: oldDistance
    };

    try {
        // Update distance
        await fetch('submit_bib.1.3.3.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(updatePayload)
        });

        // Send log entry
        await fetch('submit_bib.1.3.3.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(logPayload)
        });
    } catch (e) {
        console.warn('Sync failed:', e);
    }
}

function saveFormState() {
    try {
        const operatorName = document.getElementById('operatorName');
        const sendTo = document.getElementById('sendTo');
        const aidStation = document.getElementById('aidStation');
        if (operatorName) localStorage.setItem('operatorName', operatorName.value || '');
        if (sendTo) localStorage.setItem('sendTo', sendTo.value || '');
        if (aidStation) localStorage.setItem('aidStation', aidStation.value || '');
        console.log('Saved form state to localStorage');
    } catch (e) {
        console.error('Error saving form state:', e);
    }
}

function loadFormState() {
    try {
        const operatorName = document.getElementById('operatorName');
        const sendTo = document.getElementById('sendTo');
        const aidStation = document.getElementById('aidStation');
        if (operatorName) operatorName.value = localStorage.getItem('operatorName') || '';
        if (sendTo) sendTo.value = localStorage.getItem('sendTo') || '';
        if (aidStation) aidStation.value = localStorage.getItem('aidStation') || '';
        console.log('Loaded form state from localStorage');
    } catch (e) {
        console.error('Error loading form state:', e);
    }
}

function setCurrentTime() {
    try {
        const timeInput = document.getElementById('time');
        if (!timeInput) {
            console.error('Time input not found');
            return;
        }
        const now = new Date();
        const hours = String(now.getHours()).padStart(2, '0');
        const minutes = String(now.getMinutes()).padStart(2, '0');
        timeInput.value = `${hours}:${minutes}`;
        console.log('Time set to:', `${hours}:${minutes}`);
    } catch (e) {
        console.error('Error setting current time:', e);
    }
}

function updateSubject() {
    try {
        const eventName = document.getElementById('eventName')?.value || 'Ray Miller 50-50';
        const station = document.getElementById('aidStation')?.value || '';
        const msgNum = document.getElementById('messageNum')?.value || '1';
        const subjectInput = document.getElementById('subject');
        if (!subjectInput) {
            console.error('Subject input not found');
            return;
        }
        subjectInput.value = `${eventName} ${station} Message #${msgNum}`;
        console.log('Subject updated:', subjectInput.value);
    } catch (e) {
        console.error('Error updating subject:', e);
    }
}

function incrementMessage() {
    try {
        const msgNum = document.getElementById('messageNum');
        if (!msgNum) {
            console.error('Message number input not found');
            return;
        }
        msgNum.value = parseInt(msgNum.value) + 1;
        updateSubject();
        console.log('Message number incremented to:', messageNum.value);
    } catch (e) {
        console.error('Error incrementing message:', e);
    }
}

function checkSpelling(comment) {
    const commonTypos = {
        'anle': 'ankle',
        'recoded': 'recorded',
        'jogs': 'jugs',
        'medica': 'medic'
    };
    for (let typo in commonTypos) {
        if (comment.toLowerCase().includes(typo)) {
            return `Possible typo: "${typo}" (did you mean "${commonTypos[typo]}")? Add anyway?`;
        }
    }
    return null;
}

function saveData(isManualSave = false) {
    try {
        const data = { entries, messageLog, bibList, offlineQueue, generalComments };
        localStorage.setItem('bibData', JSON.stringify(data));
        if (isManualSave) {
            alert(`Data saved locally! (${entries.length} entries, ${messageLog.length} messages, ${bibList.length} bibs, ${offlineQueue.length} offline entries)`);
        }
        console.log('Saving data locally:', data);
    } catch (e) {
        console.error('Error saving data:', e);
        alert('Error saving data: ' + e.message);
    }
}

function sortEntries() {
    entries.sort((a, b) => {
        let aDateTime, bDateTime;
        try {
            aDateTime = new Date(`${a.date} ${a.time}`).getTime();
        } catch {
            aDateTime = NaN;
        }
        try {
            bDateTime = new Date(`${b.date} ${b.time}`).getTime();
        } catch {
            bDateTime = NaN;
        }
        if (isNaN(aDateTime) && isNaN(bDateTime)) return 0;
        if (isNaN(aDateTime)) return 1; // Invalid at end
        if (isNaN(bDateTime)) return -1;
        return bDateTime - aDateTime; // Newest first
    });
    console.log('Entries sorted newest first');
}

function clearForm() {
    try {
        document.getElementById('bibNumber').value = '';
        document.getElementById('time').value = '';
        document.getElementById('yesterday').checked = false;
        document.getElementById('comment').value = '';
        document.getElementById('eta').value = '';
        document.getElementById('generalComment').value = '';
        document.getElementById('bibInfoTable').style.display = 'none';
        setCurrentTime();
        setTimeout(() => {
            document.getElementById('bibNumber').focus();
        }, 100);  // Delay for mobile keyboard
        console.log('Form cleared');
    } catch (e) {
        console.error('Error clearing form:', e);
    }
}

function cancelEdit() {
    editingIndex = null;
    clearForm();
    const updateButton = document.getElementById('updateButton');
    if (updateButton) updateButton.style.display = 'none';
    const cancelEditButton = document.getElementById('cancelEditButton');
    if (cancelEditButton) cancelEditButton.style.display = 'none';
    const changeDistanceButton = document.getElementById('changeDistanceButton');
    if (changeDistanceButton) changeDistanceButton.style.display = 'inline-block';
    console.log('Edit canceled');
}

async function loadData() {
    loadGeneralComments();
    const loadedData = JSON.parse(localStorage.getItem('bibData') || '{}');
    bibList = loadedData.bibList || [];
    entries = loadedData.entries || [];  // Load local entries first
    offlineQueue = loadedData.offlineQueue || [];  // Load queue first
    console.log('Loaded offline queue length:', offlineQueue.length);  // Debug
    try {
        const online = await isOnline();
        if (online && offlineQueue.length > 0) {
            await syncOfflineEntries();  // Sync queue before fetching
        }
        if (!online) {
            throw new Error('Offline - loading from localStorage'); // Force fallback
        }
        const response = await fetch('load_bib.1.3.3.php?limit=500');
        if (!response.ok) {
            throw new Error(`Network response was not ok: ${response.status}`);
        }
        const data = await response.json();
        console.log('Server response:', data);
	if (Array.isArray(data)) {
	  // ✅ Only keep rows that actually have a non-empty status string
	  const filteredData = data.filter(item => item.status && item.status.trim() !== '');

	  entries = filteredData.map(item => {
	    const statusParts = item.status ? item.status.split(' | ') : [];
	    const len = statusParts.length;

	    if (len < 6) {
	      console.warn('Incomplete status string:', item.status, 'Length:', len);
	    }

	    const runner =
	      bibList.find(r => String(r.bib) === String(item.bib)) || {};

	    // Use filteredData here, NOT data, and no extra ".filter" typo
	    const latestChange = filteredData
	      .filter(
	        e =>
	          String(e.bib) === String(item.bib) &&
	          e.status &&
	          e.status.startsWith('Change Distance')
	      )
	      .sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp))[0]; // or [0]/[2] as you prefer

	    // Build Date/Time from timestamp if possible
	    let displayDate = '';
	    let displayTime = '';

	    if (item.timestamp) {
	      try {
	        const dt = new Date(item.timestamp);
	        const yyyy = dt.getFullYear();
	        const mm = String(dt.getMonth() + 1).padStart(2, '0');
	        const dd = String(dt.getDate()).padStart(2, '0');
	        const hh = String(dt.getHours()).padStart(2, '0');
	        const mi = String(dt.getMinutes()).padStart(2, '0');
	        displayDate = `${yyyy}-${mm}-${dd}`;
	        displayTime = `${hh}:${mi}`;
	      } catch (e) {
	        console.warn('Bad timestamp for item', item, e);
	      }
	    }

	// fallback: if time still empty, try statusParts[1] if it looks like HH:MM
	    const statusTimePart = statusParts[1] || '';
	    if (!displayTime && /^\d{1,2}:\d{2}$/.test(statusTimePart)) {
	      displayTime = statusTimePart;
	    }

	    // Day ("Today", etc.)
	    const displayDay = statusParts[2] || 'N/A';

	    // Distance-change detection
	    const isDistanceChangeLabel = statusTimePart.toLowerCase().includes('distance change');
	    const baseComment = statusParts[4] || '';
	    const displayComment = isDistanceChangeLabel
	      ? [statusTimePart, baseComment].filter(Boolean).join(' — ')
	      : baseComment;

	    // 🔧 Fix misalignment for short / distance-change statuses
	    let etaValue, operatorValue, eventNameValue, messageNumValue;

	    if (isDistanceChangeLabel || len <= 6) {
	      // Distance-change style status: no ETA, operator is last piece
	      etaValue = ''; // or 'N/A' if you prefer to show N/A
	      operatorValue = statusParts[len - 1] || 'N/A';

	      // Event: use whatever we know about this race, don't pull from statusParts
	      eventNameValue =
	        item.eventName ||
	        (document.getElementById('eventName')?.value) ||
	        'Ray Miller 50 50 2025';

	      // Message number not baked into this short status string
	      messageNumValue = item.message_num || '1';
	    } else {
	      // Full normal status with ETA/operator/event/messageNum at the end
	      etaValue = statusParts[5] || 'N/A';
	      operatorValue = statusParts[6] || 'N/A';
	      eventNameValue = statusParts[len - 3] || 'RayMiller-50-50';
	      messageNumValue = statusParts[len - 2] || item.message_num || '1';
	    }

	    return {
	      eventName: eventNameValue,
	      bib_number: item.bib || 'N/A',
	      action: statusParts[0] || 'N/A',
	      time: displayTime || statusTimePart || 'N/A',
	      day: displayDay,
	      station: statusParts[3] || 'N/A',
	      comment: displayComment,
	      eta: etaValue || 'N/A',
	      operator: operatorValue || 'N/A',
	      date: displayDate || item.date || 'N/A',
	      messageNum: messageNumValue,
	      first_name: item.first_name || runner.firstName || 'N/A',
	      last_name: item.last_name || runner.lastName || 'N/A',
	      age: item.age || parseInt(runner.age) || 'N/A',
	      gender: item.gender || runner.gender || 'N/A',
	      distance: latestChange ? latestChange.distance : (item.distance || runner.distance || 'N/A'),
	      previous_distance: latestChange ? latestChange.previous_distance :
	        (item.previous_distance || runner.previousDistance || 'N/A')
	    };



            });
            sortEntries(); // Robust sort
            filterBibLog();
            const entryCount = document.getElementById('entryCount');
            if (entryCount) entryCount.textContent = entries.length;
            saveData(false);
        } else {
            console.error('Received non-array data:', data);
            alert('Error: Invalid data format from server. Loading from localStorage.');
            entries = loadedData.entries || [];
            offlineQueue = loadedData.offlineQueue || [];
            sortEntries();
            filterBibLog();
            const entryCount = document.getElementById('entryCount');
            if (entryCount) entryCount.textContent = entries.length;
            console.log('Loaded data from localStorage:', loadedData);
        }
    } catch (error) {
        console.error('Error in loadData:', error);
        console.log('loadData catch, showAlerts:', showAlerts); // Debug
        if (showAlerts) alert('Offline or server error: ' + error.message);
        const loadedData = JSON.parse(localStorage.getItem('bibData') || '{}');
        entries = loadedData.entries || [];
        offlineQueue = loadedData.offlineQueue || [];
        console.log('Offline queue length on load:', offlineQueue.length);
        sortEntries(); // Robust sort
        filterBibLog();
        const entryCount = document.getElementById('entryCount');
        if (entryCount) entryCount.textContent = entries.length;
        console.log('Loaded data from localStorage:', loadedData);
    }
}

async function submitDistanceChange(bib, newDistance, oldDistance) {
    const payload = {
        bib: bib,
        distance: newDistance,
        previous_distance: oldDistance,
        action: 'distance_change',
        operator: document.getElementById('operatorName')?.value.trim() || 'Unknown'
    };

    try {
        const response = await fetch('submit_bib.1.3.3.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        });
        const result = await response.json();
        if (result.success) {
            console.log('Distance change saved to server');
        } else {
            console.warn('Server rejected distance change:', result.error);
        }
    } catch (e) {
        console.error('Failed to sync distance change:', e);
    }
}

function updateDistance() {
    const select = document.getElementById('distanceSelect');
    if (!select) return;
    const newDistance = select.value;

    try {
        const bib = document.getElementById('bibNumber').value.trim();
        if (!bib) {
            alert('No bib number entered.');
            return;
        }

        const runner = bibList.find(r => String(r.bib) === String(bib));
        if (!runner) {
            alert('Runner not found in bib list.');
            return;
        }

        const oldDistance = runner.distance || 'N/A';
        if (oldDistance === newDistance) {
            alert('Distance is already set to ' + newDistance + '.');
            closeDistancePopup();
            return;
        }

        // Update runner
        runner.previousDistance = oldDistance;
        runner.distance = newDistance;

        // Create a FAKE log entry that will show in the table
        const now = new Date();
        const time = `${String(now.getHours()).padStart(2,'0')}:${String(now.getMinutes()).padStart(2,'0')}`;
        const station = document.getElementById('aidStation').value || '';
        const operator = document.getElementById('operatorName').value.trim() || 'Unknown';

        const fakeStatus = newDistance === 'DNS' 
             ? `DNS | ${time} | Today | ${station} | Did Not Start | ${operator}`
             : `IN | ${time} | Today | ${station} | Distance changed to ${newDistance} | ${operator}`;

        // Send both: update distance AND fake log entry
        fetch('submit_bib.1.3.3.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'distance_change',
                bib: parseInt(bib),
                distance: newDistance,
                previous_distance: oldDistance
            })
        });

        fetch('submit_bib.1.3.3.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                bib: parseInt(bib),
                status: fakeStatus,
                distance: newDistance,
                previous_distance: oldDistance
            })
        });

        // Local log entry
        const entry = {
            eventName: document.getElementById('eventName').value || 'Ray Miller 50 50 2025',
            bib_number: bib,
            action: newDistance === 'DNS' ? 'DNS' : 'IN',
            time,
            day: 'Today',
            station,
            comment: newDistance === 'DNS' ? 'Did Not Start' : `Distance changed to ${newDistance}`,
            eta: '',
            operator,
            date: now.toLocaleDateString('en-US', { timeZone: 'America/Los_Angeles' }),
            messageNum: document.getElementById('messageNum').value || '1',
            first_name: runner.firstName || 'N/A',
            last_name: runner.lastName || 'N/A',
            distance: newDistance,
            previous_distance: oldDistance
        };

        entries.unshift(entry);
        sortEntries();
        filterBibLog();
        saveData();
        closeDistancePopup();
        clearForm();

    } catch (e) {
        console.error('Error:', e);
    }
}

function updateBibInfo() {
    try {
        const bib = document.getElementById('bibNumber')?.value.trim();
        const tbody = document.querySelector('#bibInfoTable tbody');
        if (!tbody) {
            console.error('Bib info table body not found');
            return;
        }
        tbody.innerHTML = '';
	// After.css — after tbody.innerHTML = ...
         const wrapper = document.querySelector('.bib-log-wrapper');
         if (wrapper) {
             wrapper.scrollTop = 0;
        }
        if (bib && bibList.length > 0) {
            const runner = bibList.find(r => String(r.bib) === String(bib));
            if (runner) {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${runner.bib}</td>
                    <td>${runner.firstName || 'N/A'}</td>
                    <td>${runner.lastName || 'N/A'}</td>
                    <td>${runner.age || 'N/A'}</td>
                    <td>${runner.gender || 'N/A'}</td>
                    <td>${runner.distance || 'N/A'}</td>
                    <td>${runner.previousDistance || 'N/A'}</td>
                `;
                tbody.appendChild(row);
                document.getElementById('bibInfoTable').style.display = 'table';
                console.log('Bib info updated:', runner);
            } else {
                document.getElementById('bibInfoTable').style.display = 'none';
                console.log('No runner found for bib:', bib);
            }
        } else {
            document.getElementById('bibInfoTable').style.display = 'none';
            console.log('Bib input empty or no bib list loaded');
        }
    } catch (e) {
        console.error('Error updating bib info:', e);
        alert('Error updating bib info: ' + e.message);
    }
}

function importBibList() {
    try {
        if (!window.FileReader) {
            alert('FileReader API not supported on this device. Try a different browser or device.');
            return;
        }
        const input = document.createElement('input');
        input.type = 'file';
        input.accept = '.csv,.xlsx';
        input.onchange = async function(event) {
            const file = event.target.files[0];
            if (!file) return;
            const reader = new FileReader();
            reader.onload = async function(e) {
                const data = e.target.result;
                const wb = XLSX.read(data, { type: 'binary' });
                const ws = wb.Sheets[wb.SheetNames[0]];
                const rows = XLSX.utils.sheet_to_json(ws, { header: 1 }).slice(1);
                bibList = [];
                rows.forEach((row, index) => {
                    if (row[0]) {
                        const bibEntry = {
                            bib: String(row[0]).trim(),
                            firstName: row[1] ? String(row[1]).trim() : '',
                            lastName: row[2] ? String(row[2]).trim() : '',
                            age: row[3] ? String(row[3]).trim() : '',
                            gender: row[4] ? String(row[4]).trim() : '',
                            distance: row[5] ? String(row[5]).trim() : 'N/A',
                            previousDistance: 'N/A'
                        };
                        bibList.push(bibEntry);
                        console.log(`Parsed bib entry ${index + 1}:`, bibEntry);
                    }
                });
                console.log(`Imported ${bibList.length} bibs`, bibList.slice(0, 3));
                alert(`Imported ${bibList.length} bibs! Changes auto-saved.`);
                saveData(false);
                updateBibInfo();
                filterBibLog();
            };
            reader.readAsBinaryString(file);
        };
        input.click();
    } catch (e) {
        console.error('Error importing bib list:', e);
        alert('Error importing bib list: ' + e.message);
    }
}

function showDistancePopup() {
    try {
        const bib = document.getElementById('bibNumber')?.value.trim();
        if (!bib) {
            alert('Enter a Bib Number to change distance.');
            return;
        }
        if (!bibList.find(r => String(r.bib) === String(bib))) {
            alert('Bib not found in imported list.');
            return;
        }
        const popup = document.getElementById('distancePopup');
        if (popup) {
            popup.style.display = 'block';
            document.getElementById('distanceSelect')?.focus();
            console.log('Distance popup shown for bib:', bib);
        } else {
            console.error('Distance popup not found');
        }
    } catch (e) {
        console.error('Error showing distance popup:', e);
        alert('Error showing distance popup: ' + e.message);
    }
}



function closeDistancePopup() {
    try {
        const popup = document.getElementById('distancePopup');
        if (popup) {
            popup.style.display = 'none';
            document.getElementById('bibNumber').focus();
            console.log('Distance popup closed');
        }
    } catch (e) {
        console.error('Error closing distance popup:', e);
    }
}

function safeString(str) {
    return (str || '').replace(/\|/g, '-');
}


function filterBibLog() {
    const table = document.getElementById('bibLogTable');
    if (!table) {
        console.error('Bib Log table not found in DOM');
        setTimeout(filterBibLog, 100); // Retry after 100ms
        return;
    }
    let tbody = table.getElementsByTagName('tbody')[0];
    if (!tbody) {
        console.warn('Bib Log table body not found, creating one');
        tbody = document.createElement('tbody');
        table.appendChild(tbody);
    }
    const searchInput = document.getElementById('logSearch')?.value.toLowerCase() || '';
    const filteredEntries = entries.filter(entry => {
        return Object.values(entry).some(val =>
            String(val).toLowerCase().includes(searchInput)
        );
    }).slice(0, 100); // Limit to 50 for performance
    tbody.innerHTML = '';
    filteredEntries.forEach(entry => {
        const row = tbody.insertRow();
//        row.insertCell(0).textContent = entry.eventName; // Ray Miller 5050
        row.insertCell(0).textContent = entry.bib_number;
        row.insertCell(1).textContent = entry.action;
        row.insertCell(2).textContent = entry.time;
        row.insertCell(3).textContent = entry.day;
        row.insertCell(4).textContent = entry.station;
        row.insertCell(5).textContent = entry.comment;
        row.insertCell(6).textContent = entry.eta; // N/A or actual ETA
        row.insertCell(7).textContent = entry.operator; // Todd Sept 13 1620
        row.insertCell(8).textContent = entry.date; // 9/13/2025
        row.insertCell(9).textContent = entry.first_name;
        row.insertCell(10).textContent = entry.last_name;
        row.insertCell(11).textContent = entry.age;
        row.insertCell(12).textContent = entry.gender;
        row.insertCell(13).textContent = entry.distance;
        row.insertCell(14).textContent = entry.previous_distance;
        const editCell = row.insertCell(15);
        editCell.innerHTML = '<a href="#" onclick="editEntry(\'' + entry.bib_number + '\')">Edit</a>';
    });
    console.log('Bib Log filtered, rows:', filteredEntries.length);
    const bibLogTable = document.getElementById('bibLogTable');
   if (bibLogTable) {
    bibLogTable.style.display = 'table';  // Force show the table after update
    console.log('Table display set to visible');
  }
}

function sortBibTable(colIndex) {
  try {
    const columns = [
      'bib_number', 'action', 'time', 'day', 'station',   // remove from from  'eventName',
      'comment', 'eta', 'operator', 'first_name', 'last_name',
      'age', 'gender', 'distance', 'previous_distance'
    ];
    const column = columns[colIndex];   // <-- add this line

    bibSortDirection[colIndex] *= -1;
    currentSortColumn = colIndex;
    currentSortDirection = bibSortDirection[colIndex];

    const sortedData = [...entries];
    sortedData.sort((a, b) => {
      let aValue = a[column] || '';
      let bValue = b[column] || '';

      if (['first_name', 'last_name', 'age', 'gender', 'distance', 'previous_distance'].includes(column)) {
        const runnerA = bibList.find(r => String(r.bib) === String(a.bib_number)) || {};
        const runnerB = bibList.find(r => String(r.bib) === String(b.bib_number)) || {};
        aValue = a[column] || runnerA[column.replace('first_name', 'firstName').replace('last_name', 'lastName').replace('previous_distance', 'previousDistance')] || 'N/A';
        bValue = b[column] || runnerB[column.replace('first_name', 'firstName').replace('last_name', 'lastName').replace('previous_distance', 'previousDistance')] || 'N/A';
      }

      if (['bib_number', 'age'].includes(column)) {
        aValue = aValue === 'N/A' ? -Infinity : parseFloat(aValue) || aValue;
        bValue = bValue === 'N/A' ? -Infinity : parseFloat(bValue) || bValue;
      } else if (column === 'time') {
        aValue = aValue === 'N/A' ? NaN : new Date(`1970-01-01T${aValue}Z`).getTime();
        bValue = bValue === 'N/A' ? NaN : new Date(`1970-01-01T${bValue}Z`).getTime();
        if (isNaN(aValue) && isNaN(bValue)) return 0;
        if (isNaN(aValue)) return 1;
        if (isNaN(bValue)) return -1;
      } else if (column === 'date') {
        aValue = aValue === 'N/A' ? NaN : new Date(aValue).getTime();
        bValue = bValue === 'N/A' ? NaN : new Date(bValue).getTime();
        if (isNaN(aValue) && isNaN(bValue)) return 0;
        if (isNaN(aValue)) return 1;
        if (isNaN(bValue)) return -1;
      }

      if (aValue === bValue) return 0;
      if (aValue === '' || aValue === -Infinity) return 1;
      if (bValue === '' || bValue === -Infinity) return -1;
      return currentSortDirection * (aValue > bValue ? 1 : -1);
    });

    // Replace the global entries with the sorted data if needed
    entries = sortedData;
    filterBibLog();
    console.log('Bib table sorted by column', colIndex, 'direction:', currentSortDirection);
  } catch (e) {
    console.error('Error sorting Bib table:', e);
    alert('Error sorting Bib table: ' + e.message);
  }
}


function sortMessageLogTable(colIndex) {
    try {
        const columns = ['messageNum', 'action', 'timestamp'];
        const column = columns[colIndex];
        messageSortDirection[colIndex] *= -1;
        const sortedData = [...messageLog];
        sortedData.sort((a, b) => {
            let aValue = a[column] || '';
            let bValue = b[column] || '';
            if (column === 'messageNum') {
                aValue = parseFloat(aValue) || aValue;
                bValue = parseFloat(bValue) || bValue;
            } else if (column === 'timestamp') {
                aValue = new Date(aValue).getTime();
                bValue = new Date(bValue).getTime();
            }
            if (aValue === bValue) return 0;
            return messageSortDirection[colIndex] * (aValue > bValue ? 1 : -1);
        });
        const tbody = document.querySelector('#messageLogTable tbody');
        if (!tbody) {
            console.error('Message Log table body not found');
            return;
        }
        tbody.innerHTML = '';
        sortedData.forEach((m, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${m.messageNum}</td>
                <td>${m.action}</td>
                <td>${m.timestamp}</td>
                <td><button class="edit-button" onclick="showMessageDetails(${index})">Show Entries</button></td>
            `;
            tbody.appendChild(row);
        });
        console.log('Message Log table sorted by column', colIndex, 'direction:', messageSortDirection[colIndex]);
    } catch (e) {
        console.error('Error sorting Message Log table:', e);
        alert('Error sorting Message Log table: ' + e.message);
    }
}

function editEntry(bib) {
    try {
        const entry = entries.find(e => e.bib_number === bib);
        if (!entry) {
            console.error('Entry not found for bib:', bib);
            return;
        }
        editingIndex = entries.indexOf(entry);
        document.getElementById('bibNumber').value = entry.bib_number;
        document.getElementById('time').value = entry.time;
        document.getElementById('yesterday').checked = entry.day === 'Yesterday';
        document.getElementById('aidStation').value = entry.station;
        document.getElementById('comment').value = entry.comment;
        document.getElementById('eta').value = entry.eta;
        document.getElementById('operatorName').value = entry.operator;
        document.getElementById('messageNum').value = entry.messageNum;
        document.getElementById('eventName').value = entry.eventName;
        const updateButton = document.getElementById('updateButton');
        if (updateButton) updateButton.style.display = 'inline-block';
        const cancelEditButton = document.getElementById('cancelEditButton');
        if (cancelEditButton) cancelEditButton.style.display = 'inline-block';
        const changeDistanceButton = document.getElementById('changeDistanceButton');
        if (changeDistanceButton) changeDistanceButton.style.display = 'none';
        console.log('Editing entry:', entry);
    } catch (e) {
        console.error('Error editing entry:', e);
        alert('Error editing entry: ' + e.message);
    }
}

async function updateEntry(action) {
    try {
        const entry = entries[editingIndex];
        if (!entry) {
            console.error('Entry not found for update');
            return;
        }
        entry.action = action;
        entry.time = document.getElementById('time').value;
        entry.day = document.getElementById('yesterday').checked ? 'Yesterday' : 'Today';
        entry.station = document.getElementById('aidStation').value;
        entry.comment = safeString(document.getElementById('comment').value);
        entry.eta = safeString(document.getElementById('eta').value);
        entry.operator = safeString(document.getElementById('operatorName').value.trim());
        entry.messageNum = document.getElementById('messageNum').value;
        entry.eventName = document.getElementById('eventName').value;
        entry.date = new Date().toLocaleDateString('en-US', { timeZone: 'America/Los_Angeles' });
        const online = await isOnline();
        if (online) {
            const payload = {
                bib: entry.bib_number,
                status: [
                    safeString(entry.action),
                    safeString(entry.time),
                    safeString(entry.day),
                    safeString(entry.station),
                    safeString(entry.comment),
                    safeString(entry.eta),
                    safeString(entry.operator),
                    safeString(entry.eventName),
                    safeString(entry.messageNum),
                    safeString(entry.date)
                ].join(' | '),
                first_name: entry.first_name,
                last_name: entry.last_name,
                age: entry.age,
                gender: entry.gender,
                distance: entry.distance,
                previous_distance: entry.previous_distance
            };
            console.log('POST payload for update:', payload);
            const response = await fetch('submit_bib.1.3.3.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            const result = await response.json();
            if (result.success) {
                console.log('Entry updated:', payload);
                if (showAlerts) alert('Entry updated successfully!');
                await loadData();
            } else {
                console.error('Failed to update entry:', result.error);
                if (showAlerts) alert('Failed to update entry: ' + result.error + '. Changes saved locally.');
                saveData(false);
            }
        } else {
            if (showAlerts) alert('No internet connection. Changes saved locally.');
            saveData(false);
        }
        editingIndex = null;
        clearForm();
        sortEntries();
        filterBibLog();
    } catch (e) {
        console.error('Error updating entry:', e);
        if (showAlerts) alert('Error updating entry: ' + e.message + '. Changes saved locally.');
        saveData(false);
    } finally {
        const updateButton = document.getElementById('updateButton');
        if (updateButton) updateButton.style.display = 'none';
        const cancelEditButton = document.getElementById('cancelEditButton');
        if (cancelEditButton) cancelEditButton.style.display = 'none';
        const changeDistanceButton = document.getElementById('changeDistanceButton');
        if (changeDistanceButton) changeDistanceButton.style.display = 'inline-block';
        isSubmitting = false;
    }
}

async function addEntry(action) {
    try {
        if (isSubmitting) return;
        isSubmitting = true;

        let bibInput = document.getElementById('bibNumber')?.value.trim() || (action === 'GENERAL' ? 'N/A' : '');
        if (!bibInput && action !== 'GENERAL') {
            if (showAlerts) alert('Please enter a Bib Number.');
            return;
        }

        const bibs = bibInput.split(',').map(b => b.trim()).filter(b => b);
        if (bibs.length > 1 && action === 'GENERAL') {
            if (showAlerts) alert('Multiple bibs not supported for GENERAL comments.');
            return;
        }
        if (bibs.length > 1 && editingIndex !== null) {
            if (showAlerts) alert('Multiple bibs not supported while editing.');
            return;
        }

        const commentInput = action === 'GENERAL' ? 'generalComment' : 'comment';
        const comment = safeString(document.getElementById(commentInput)?.value.trim() || '');
        const spellingError = checkSpelling(comment);
        if (spellingError && showAlerts && !confirm(spellingError)) {
            return;
        }

        // Shared values for all bibs
        const time = document.getElementById('time')?.value || '';
        const day = document.getElementById('yesterday')?.checked ? 'Yesterday' : 'Today';
        const station = document.getElementById('aidStation')?.value || '';
        const eta = safeString(document.getElementById('eta')?.value || 'N/A');
        const operator = safeString(document.getElementById('operatorName')?.value.trim() || '');
        const eventName = document.getElementById('eventName')?.value || 'Ray Miller 50-50';
        const messageNum = document.getElementById('messageNum')?.value || '1';
        const date = new Date().toLocaleDateString('en-US', { timeZone: 'America/Los_Angeles' });

        for (const bib of bibs) {
            let runner = bibList.find(r => String(r.bib) === String(bib)) || {};
            const entry = {
                eventName,
                bib_number: bib,
                action: action === 'DNF' ? 'DNF' : action,
                time,
                day,
                station,
                comment,
                eta,
                operator,
                date,
                messageNum,
                first_name: runner.firstName || 'N/A',
                last_name: runner.lastName || 'N/A',
                age: runner.age || 'N/A',
                gender: runner.gender || 'N/A',
                distance: runner.distance || 'N/A',
                previous_distance: runner.previousDistance || 'N/A'
            };

            entries.unshift(entry);

            const online = await isOnline();
            if (online) {
                const payload = {
                    bib: entry.bib_number,
                    status: [
                        safeString(entry.action),
                        safeString(entry.time),
                        safeString(entry.day),
                        safeString(entry.station),
                        safeString(entry.comment),
                        safeString(entry.eta),
                        safeString(entry.operator),
                        safeString(entry.eventName),
                        safeString(entry.messageNum),
                        safeString(entry.date)
                    ].join(' | '),
                    first_name: entry.first_name,
                    last_name: entry.last_name,
                    age: entry.age,
                    gender: entry.gender,
                    distance: entry.distance,
                    previous_distance: entry.previous_distance
                };

                console.log('POST payload for add:', payload);

                const response = await fetch('submit_bib.1.3.3.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(payload)
                });

                const result = await response.json();
                if (result.success) {
                    console.log('Entry added:', payload);
                    if (showAlerts) alert('Entry added successfully!');
                    await loadData();
                } else {
                    console.error('Failed to add entry:', result.error);
                    if (showAlerts) alert('Failed to add entry: ' + result.error + '. Saved locally and queued.');
                    offlineQueue.push(entry);
                    saveData(false);
                }
            } else {
                if (showAlerts) alert('No internet connection. Entry saved locally and queued.');
                offlineQueue.push(entry);
                saveData(false);
            }
        }

        clearForm();
        sortEntries();
        filterBibLog();
        const entryCount = document.getElementById('entryCount');
        if (entryCount) entryCount.textContent = entries.length;

    } catch (e) {
        console.error('Error adding entry:', e);
        if (showAlerts) alert('Error adding entry: ' + e.message);
    } finally {
        isSubmitting = false;
    }
}

function toggleBibLog() {
    const bibLogTable = document.getElementById('bibLogTable');
    const wrapper = document.querySelector('.bib-log-wrapper');

    if (!bibLogTable || !wrapper) {
        console.error('Bib Log table or wrapper not found');
        setTimeout(toggleBibLog, 100);
        return;
    }

    const wasHidden = bibLogTable.style.display === 'none' || bibLogTable.style.display === '';

    // Show table
    bibLogTable.style.display = 'table';

    if (wasHidden) {
        setTimeout(() => {
            filterBibLog();           // Re first
            wrapper.scrollTop = 0;    // THEN SCROLL TO TOP
            console.log('Bib Log shown and scrolled to TOP');
        }, 10);
    } else {
        console.log('Bib Log hidden');
    }
}

function toggleMessageLog() {
    const messageLogTable = document.getElementById('messageLogTable');
    if (messageLogTable) {
        const currentDisplay = messageLogTable.style.display;
        console.log('Attempting to toggle Message Log');
        console.log('Current display:', currentDisplay);
        messageLogTable.style.display = currentDisplay === 'none' ? 'table' : 'none';
        console.log('Message Log toggled to:', messageLogTable.style.display);
        if (messageLogTable.style.display === 'table') {
            setTimeout(updateMessageLogTable, 0); // Ensure table is populated
        }
    } else {
        console.error('Message Log table not found in DOM');
        setTimeout(toggleMessageLog, 100); // Retry
    }
}

function exportCSV() {
    const filtered = [...entries].sort((a, b) => new Date(b.timestamp || b.date) - new Date(a.timestamp || a.date));
    const csvContent = [
        'Bib #,Action,Time,Day,Station,Comment,ETA,Operator,Date,Event,First Name,Last Name,Age,Gender,Distance,Previous Distance',
        ...filtered.map(e => [
            `"${e.bib_number}"`,
            `"${e.action}"`,
            `"${e.time}"`,
            `"${e.day}"`,
            `"${e.station}"`,
            `"${e.comment}"`,
            `"${e.eta}"`,
            `"${e.operator}"`,
            `"${e.date}"`,
            `"${e.eventName}"`,
            `"${e.first_name}"`,
            `"${e.last_name}"`,
            `"${e.age}"`,
            `"${e.gender}"`,
            `"${e.distance}"`,
            `"${e.previous_distance}"`
        ].join(','))
    ].join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `bib_log_${new Date().toISOString().replace(/[:.]/g, '-')}.csv`;
    a.click();
    window.URL.revokeObjectURL(url);
    console.log('CSV exported, entries:', filtered.length);
}

// ===============================
// BIB Winlink TXT Export (BatchCore v2)
// ===============================
function exportBibWinlinkTxt_v2() {
  // 1) Ask BatchCore for a BIB batch using the "winlink" channel
  const batch = TVEMC_BatchCore.createBibBatchForExport("winlink");

  // If no new rows, bail out cleanly
  if (!batch || !batch.records || batch.records.length === 0) {
    alert("No new bib entries to export for Winlink.");
    return;
  }

  // 2) Build the header line for Winlink / radio logs
  const msgNumPadded = String(batch.message_number || 0).padStart(3, "0");
  const headerLine =
    `MSG:${msgNumPadded} ` +
    `${batch.event_id || "EVENT"} ` +
    `${batch.aid_station_name || "STATION"} ` +
    `OP:${batch.operator || "OPERATOR"} ` +
    `${batch.timestamp_local || ""} ` +
    `BIB EXPORT`;

  // 3) Column headers (same columns as CSV export)
  const columnHeader = [
    "Bib #",
    "Action",
    "Time",
    "Day",
    "Station",
    "Comment",
    "ETA",
    "Operator",
    "Date",
    "Event",
    "First Name",
    "Last Name",
    "Age",
    "Gender",
    "Distance",
    "Previous Distance"
  ].join(",");

  // 4) Convert each batch record into a CSV-style line
  const bodyLines = batch.records.map(e => [
    e.bib_number ?? "",
    e.action ?? "",
    e.time ?? "",
    e.day ?? "",
    e.station ?? "",
    e.comment ?? "",
    e.eta ?? "",
    e.operator ?? "",
    e.date ?? "",
    e.eventName ?? "",
    e.first_name ?? "",
    e.last_name ?? "",
    e.age ?? "",
    e.gender ?? "",
    e.distance ?? "",
    e.previous_distance ?? ""
  ].map(value => `"${String(value).replace(/"/g, '""')}"`).join(","));

  // 5) Full text content for the file: header + columns + rows
  const txtContent = [headerLine, columnHeader, ...bodyLines].join("\n");

  // 6) Build a nice filename using BatchCore helper
  const filename = TVEMC_BatchCore.filenameForBatch(batch, "txt");

  // 7) Trigger a download as .txt (you already have downloadFile for CSV)
  downloadFile(filename, txtContent, "text/plain");

  // 8) Let the operator know which message number this was
  alert(`BIB Winlink TXT Exported\nMessage #${batch.message_number}`);
}


function updateMessageLogTable() {
    try {
        console.log('Updating Message Log table, entries:', messageLog);
        const tbody = document.querySelector('#messageLogTable tbody');
        if (!tbody) {
            console.error('Message log table body not found');
            return;
        }
        tbody.innerHTML = '';
        messageLog.forEach((m, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${m.messageNum}</td>
                <td>${m.action}</td>
                <td>${m.timestamp}</td>
                <td><button class="edit-button" onclick="showMessageDetails(${index})">Show Entries</button></td>
            `;
            tbody.insertBefore(row, tbody.firstChild);
        });
        console.log('Message Log table updated, rows:', tbody.children.length);
    } catch (e) {
        console.error('Error updating Message Log table:', e);
        alert('Error updating Message Log table: ' + e.message);
    }
}

function showMessageDetails(index) {
    try {
        const m = messageLog[index];
        let details = `Message #${m.messageNum} (${m.action}, ${m.timestamp})\n\n`;
        details += 'Bib #\tAction\tTime\tDay\tStation\tComment\tETA\tOperator\tEvent\tFirst Name\tLast Name\tAge\tGender\tDistance\tPrevious Distance\n';
        m.entries.forEach(e => {
            const runner = bibList.find(r => String(r.bib) === String(e.bib_number)) || {
                firstName: e.first_name || 'N/A',
                lastName: e.last_name || 'N/A',
                age: e.age || 'N/A',
                gender: e.gender || 'N/A',
                distance: e.distance || 'N/A',
                previousDistance: e.previous_distance || 'N/A'
            };
            details += `${e.bib_number || 'N/A'}\t${e.action === 'DROP' ? 'DNF' : e.action}\t${e.time || 'N/A'}\t${e.day || 'N/A'}\t${e.station || 'N/A'}\t${e.comment || 'N/A'}\t${e.eta || 'N/A'}\t${e.operator || 'N/A'}\t${e.eventName || 'N/A'}\t${runner.firstName || 'N/A'}\t${runner.lastName || 'N/A'}\t${runner.age || 'N/A'}\t${runner.gender || 'N/A'}\t${runner.distance || 'N/A'}\t${runner.previousDistance || 'N/A'}\n`;
        });
        console.log('Showing details for Message #', m.messageNum);
        alert(details);
    } catch (e) {
        console.error('Error showing message details:', e);
        alert('Error showing message details: ' + e.message);
    }
}

function clearLog() {
    try {
        if (!confirm('Are you sure you want to clear all entries, message log, and bib list? This will reset the current session but preserve previously saved data for restoration with Load Data.')) {
            console.log('Clear canceled at first prompt');
            return;
        }
        const confirmation = prompt('Type "CLEAR" to confirm clearing all current data:');
        if (confirmation && confirmation.toUpperCase() === 'CLEAR') {
            entries = [];
            messageLog = [];
            bibList = [];
            offlineQueue = [];
            const bibTbody = document.querySelector('#bibTable tbody');
            if (bibTbody) bibTbody.innerHTML = '';
            const messageTbody = document.querySelector('#messageLogTable tbody');
            if (messageTbody) messageTbody.innerHTML = '';
            const bibInfoTbody = document.querySelector('#bibInfoTable tbody');
            if (bibInfoTbody) bibInfoTbody.innerHTML = '';
            const bibInfoTable = document.getElementById('bibInfoTable');
            if (bibInfoTable) bibInfoTable.style.display = 'none';
            const entryCount = document.getElementById('entryCount');
            if (entryCount) entryCount.textContent = 0;
            const generalComment = document.getElementById('generalComment');
            if (generalComment) generalComment.value = '';
            console.log('Logs cleared after double confirmation');
            alert('Logs cleared successfully! Use Load Data to restore previously saved data.');
            saveData(false);
        } else {
            console.log('Clear canceled at second prompt, input:', confirmation);
            alert('Clear canceled: You must type "CLEAR" to confirm.');
        }
    } catch (e) {
        console.error('Error clearing log:', e);
        alert('Error clearing log: ' + e.message);
    }
}

function clearGeneralMessages() {
    const code = prompt("Enter authorization code to CLEAR ALL GENERAL MESSAGES:");

    // Only allow exact match
    if (code !== "CLEAR123") {
        alert("Incorrect code. Access denied.");
        return;
    }

    if (!confirm("Clear ALL general messages? This cannot be undone.")) {
        return;
    }

    // Show loading
    const btn = document.getElementById('btnClearGeneralMessages');
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = 'Clearing...';

    fetch('submit_general_comment.php', {
        method: 'POST',
        credentials: 'include',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=clear_general_messages'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert("All general messages have been cleared.");
            loadGeneralComments(); // Refresh the list
        } else {
            alert("Error: " + (data.error || "Failed to clear messages."));
        }
    })
    .catch(err => {
        console.error("Fetch error:", err);
        alert("Network error. Check console (F12).");
    })
    .finally(() => {
        // Restore button
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
}

function importCSV() {
    try {
        if (!window.FileReader) {
            alert('FileReader API not supported on this device. Try a different browser or device.');
            return;
        }
        const input = document.createElement('input');
        input.type = 'file';
        input.accept = '.csv,.xlsx';
        input.onchange = async function(event) {
            const file = event.target.files[0];
            if (!file) return;
            const reader = new FileReader();
            reader.onload = async function(e) {
                const data = e.target.result;
                const wb = XLSX.read(data, { type: 'binary' });
                const ws = wb.Sheets[wb.SheetNames[0]];
                const rows = XLSX.utils.sheet_to_json(ws, { header: 1 }).slice(1);
                let added = 0;
                const newEntries = [];
                const currentMessageNum = document.getElementById('messageNum')?.value;
                const online = await isOnline();
                for (const row of rows) {
                    if (row[0] && row[1] && row[4] && row[7]) {
                        const action = row[1] === 'DROP' ? 'DNF' : row[1];
                        const entry = {
                            bib_number: String(row[0]).trim(),
                            action: action,
                            time: row[2] || 'N/A',
                            day: row[3] || 'N/A',
                            station: row[4],
                            comment: row[5] || '',
                            eta: row[6] || 'N/A',
                            operator: row[7].replace(/[.,]/g, '').trim(),
                            eventName: row[8] || 'Ray Miller 50-50',
                            messageNum: currentMessageNum,
                            date: row[9] || new Date().toLocaleDateString('en-US', { timeZone: 'America/Los_Angeles' }),
                            first_name: row[10] || '',
                            last_name: row[11] || '',
                            age: parseInt(row[12]) || 'N/A',
                            gender: row[13] || 'N/A',
                            distance: row[14] || 'N/A',
                            previous_distance: row[15] || 'N/A'
                        };
                        const duplicate = entries.find(e =>
                            String(e.bib_number) === String(entry.bib_number) &&
                            e.action === entry.action &&
                            e.time === entry.time &&
                            e.day === entry.day &&
                            e.station === entry.station
                        );
                        if (duplicate) continue;
                        if (online) {
                            const payload = {
                                bib: entry.bib_number,
                                status: [
                                    entry.action,
                                    entry.time,
                                    entry.day,
                                    entry.station,
                                    entry.comment,
                                    entry.eta,
                                    entry.operator,
                                    entry.eventName,
                                    entry.messageNum,
                                    entry.date
                                ].filter(Boolean).join(' | '),
                                first_name: entry.first_name,
                                last_name: entry.last_name,
                                age: entry.age,
                                gender: entry.gender,
                                distance: entry.distance,
                                previous_distance: entry.previous_distance
                            };
                            console.log('POST payload for import:', payload);
                            const response = await fetch(`submit_bib.1.3.3.php`, {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify(payload)
                            });
                            if (!response.ok) {
                                throw new Error(`HTTP error! Status: ${response.status}`);
                            }
                            const data = await response.json();
                            if (data.success) {
                                newEntries.unshift(entry);  //chge 11-15 newest on to for scroll bar
                                added++;
                            } else {
                                throw new Error(data.error || 'Import failed');
                            }
                        } else {
                            offlineQueue.push(entry);
                            newEntries.push(entry);
                        }
                    }
                }
                newEntries.forEach(entry => entries.unshift(entry));
                const loadedData = JSON.parse(localStorage.getItem('bibData') || '{}');
                loadedData.entries = entries;
                loadedData.offlineQueue = offlineQueue;
                localStorage.setItem('bibData', JSON.stringify(loadedData));
		entries = loadedData.entries;  // Ensure global is in sync
		offlineQueue = loadedData.offlineQueue;
                sortEntries(); // Robust sort
                filterBibLog();
                const entryCount = document.getElementById('entryCount');
                if (entryCount) entryCount.textContent = entries.length;
                alert(`Imported ${added} entries successfully! ${newEntries.length - added} entries queued for sync.`);
                await loadData();
            };
            reader.readAsBinaryString(file);
        };
        input.click();
    } catch (e) {
        console.error('Error importing CSV:', e);
        alert('Error importing CSV: ' + e.message);
    }
}

function resetToNewest() {
  try {
    bibSortDirection = Array(15).fill(1);
    messageSortDirection = Array(3).fill(1);
    currentSortColumn = null;
    currentSortDirection = 1;
    sortEntries(); // Use robust sort instead of basic
    filterBibLog();
    console.log('Reset to newest-first order');
  } catch (e) {
    console.error('Error resetting to newest:', e);
    alert('Error resetting to newest: ' + e.message);
  }
}

async function syncOfflineEntries() {
  try {
    const online = await isOnline();
    if (!online) {
      if (showAlerts) {
        alert('No internet connection. Cannot sync offline entries yet.');
      }
      console.log('Sync attempted while offline');
      return;
    }

    if (offlineQueue.length === 0) {
      if (showAlerts) {
        alert('No offline entries to sync.');
      }
      console.log('No offline entries to sync');
      return;
    }

    let syncedCount = 0;
    const failedEntries = [];

    for (let i = offlineQueue.length - 1; i >= 0; i--) {
      const entry = offlineQueue[i];

      const payload = {
        bib: entry.bib_number,
        status: [
          safeString(entry.action),
          safeString(entry.time),
          safeString(entry.day),
          safeString(entry.station),
          safeString(entry.comment),
          safeString(entry.eta),
          safeString(entry.operator),
          safeString(entry.eventName),
          safeString(entry.messageNum),
          safeString(entry.date),
        ].join(' | '),
        first_name: entry.first_name,
        last_name: entry.last_name,
        age: entry.age,
        gender: entry.gender,
        distance: entry.distance,
        previous_distance: entry.previous_distance,
      };

      console.log('Syncing offline entry:', payload);

      const response = await fetch('submit_bib.1.3.3.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
      });

      const result = await response.json();

      if (result.success) {
        // Add to main entries if not already there (to avoid duplicates)
        const duplicate = entries.find(e =>
          e.bib_number === entry.bib_number &&
          e.action === entry.action &&
          e.time === entry.time &&
          e.day === entry.day &&
          e.station === entry.station
        );

        if (!duplicate) {
          entries.unshift(entry);
        }

        // Remove from queue
        offlineQueue.splice(i, 1);
        syncedCount++;
      } else {
        console.error('Failed to sync entry:', result.error);

        // If this entry can never be valid (e.g., missing bib),
        // drop it from the queue so it stops retrying forever
        if (result.error === 'Bib required') {
          offlineQueue.splice(i, 1);
        } else {
          failedEntries.push(entry);
        }
      }
    }

    // Update localStorage with the new state
    saveData(false);

    // Refresh UI
    sortEntries();
    filterBibLog();
    const entryCount = document.getElementById('entryCount');
    if (entryCount) entryCount.textContent = entries.length;

    if (syncedCount > 0) {
      if (showAlerts) {
        alert(`Synced ${syncedCount} entries successfully! ${failedEntries.length} failed and remain queued.`);
      }
    } else {
      if (showAlerts) {
        alert('No entries were synced. Check console for errors.');
      }
    }

    console.log('Sync complete. Remaining in queue:', offlineQueue.length);
  } catch (error) {
    console.error('Error during sync:', error);
    if (showAlerts) {
      alert('Error during sync: ' + error.message + '. Some entries may remain queued.');
    }
  }
}


    document.addEventListener('DOMContentLoaded', () => {
    try {
        console.log('DOM fully loaded');
        const fields = ['operatorName', 'sendTo', 'aidStation'];
        fields.forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('input', saveFormState);
                element.addEventListener('change', saveFormState);
                console.log(`Added saveFormState listener for ${id}`);
            }
        });
        const subjectTriggers = ['eventName', 'aidStation', 'messageNum'];
        subjectTriggers.forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('input', updateSubject);
                element.addEventListener('change', updateSubject);
                console.log(`Added updateSubject listener for ${id}`);
            }
        });
        const fastMode = document.getElementById('fastMode');
        if (fastMode) {
            fastMode.addEventListener('change', function() {
                try {
                    const timeInput = document.getElementById('time');
                    const yesterdayInput = document.getElementById('yesterday');
                    if (this.checked) {
                        if (timeInput) timeInput.tabIndex = -1;
                        if (yesterdayInput) yesterdayInput.tabIndex = -1;
                        setCurrentTime();
                    } else {
                        if (timeInput) timeInput.tabIndex = 10;
                        if (yesterdayInput) yesterdayInput.tabIndex = 11;
                    }
                    console.log('Fast Mode toggled:', this.checked);
                } catch (e) {
                    console.error('Error toggling Fast Mode:', e);
                    if (showAlerts) alert('Error toggling Fast Mode: ' + e.message);
                }
            });
            console.log('Added fastMode listener');
        }
        const highContrast = document.getElementById('highContrast');
        if (highContrast) {
            highContrast.addEventListener('change', function() {
                try {
                    document.body.classList.toggle('high-contrast', this.checked);
                    console.log('High Contrast Mode:', this.checked);
                } catch (e) {
                    console.error('Error toggling High Contrast:', e);
                    if (showAlerts) alert('Error toggling High Contrast: ' + e.message);
                }
            });
            console.log('Added highContrast listener');
        }
        const actionButtons = document.querySelectorAll(
         '#raceForm .in-button, #raceForm .out-button, #raceForm .dnf-button, #raceForm .note-button, #raceForm .add-general-comment-button'
       );
       actionButtons.forEach(btn => {
        btn.addEventListener('click', (event) => {
          event.preventDefault();
          const actionLabel = btn.textContent.trim();
          const isGeneral = (actionLabel === 'Add General Comment');

         // If it's the General Comment button, always use the general comment path
         if (isGeneral) {
           addGeneralComment();
           console.log('Action button clicked: Add General Comment (routed to addGeneralComment)');
           return;
         }

         // For all the normal bib actions (IN / OUT / DNF / NOTE),
         // keep using updateEntry/addEntry like before
         if (editingIndex !== null) {
          updateEntry(actionLabel);
         } else {
           addEntry(actionLabel);
         }
         console.log(`Action button clicked: ${actionLabel}`);
       });
     });


        console.log('Added action button listeners');
        const logSearch = document.getElementById('logSearch');
        if (logSearch) {
            logSearch.addEventListener('input', filterBibLog);
            console.log('Added search listener');
        }
        loadFormState();
        setCurrentTime();
        updateSubject();
        loadData();
        const showAlertsCheckbox = document.getElementById('showAlerts');
        if (showAlertsCheckbox) {
            showAlertsCheckbox.checked = showAlerts;
            showAlertsCheckbox.addEventListener('change', function() {
                showAlerts = this.checked;
                localStorage.setItem('showAlerts', showAlerts);
            });
        }

    } catch (e) {
        console.error('Error initializing page:', e);
        if (showAlerts) alert('Error initializing page: ' + e.message);
    }
});


// === GENERAL MESSAGE LOG SYSTEM ===
let generalComments = [];

async function addGeneralComment() {
  // debounce guard
  if (window.__addingGeneralComment) {
    console.log('Duplicate general comment prevented');
    return;
  }
  window.__addingGeneralComment = true;
  setTimeout(() => { window.__addingGeneralComment = false; }, 1500);

  // collect form values
  const commentInput = document.getElementById('generalComment');
  const comment = commentInput.value.trim();
  if (!comment) {
    if (showAlerts) alert('Please enter a general comment.');
    return;
  }

  const aidStation = document.getElementById('aidStation').value || '';
  const operator   = document.getElementById('operatorName').value.trim() || 'Unknown';
  const eventName  = document.getElementById('eventName').value || 'Ray Miller 50-50';

  // message number handling
  if (eventName !== localStorage.getItem('lastEvent')) {
    localStorage.setItem('msgCounter', 0);
    localStorage.setItem('lastEvent', eventName);
  }
  const nextMsg = parseInt(localStorage.getItem('msgCounter') || '0', 10) + 1;
  localStorage.setItem('msgCounter', nextMsg);
  const messageNum = nextMsg;

  const timestamp = new Date().toLocaleString('en-US', { timeZone: 'America/Los_Angeles' });

  const entry = { aidStation, comment, operator, eventName, messageNum, timestamp };
  generalComments.unshift(entry);

  // send to server
  const online = await isOnline();
  if (online) {
    try {
      const response = await fetch('submit_general_comment.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ aidStation, comment, operator, eventName, messageNum })
      });
      const result = await response.json();
      if (!result.success) throw new Error(result.error);
    } catch (e) {
      offlineQueue.push({ type: 'GENERAL', ...entry });
      if (showAlerts) alert('Offline. General comment queued.');
    }
  } else {
    offlineQueue.push({ type: 'GENERAL', ...entry });
    if (showAlerts) alert('Offline. General comment saved locally.');
  }

  commentInput.value = '';
  updateGeneralMessageLog();
  saveData(false);
}

function updateGeneralMessageLog() {
  const tbody = document.querySelector('#generalMessageLogTable tbody');
  const thead = document.querySelector('#generalMessageLogTable thead');

  if (!tbody || !thead) return;

  // Ensure the table header has the correct order
  const headerRow = thead.querySelector('tr');

  // Add the headers in the correct order
  const headers = ['Aid Station', 'Msg #', 'Comment', 'Operator', 'Time'];

  headerRow.innerHTML = ''; // Clear existing headers
  headers.forEach(headerText => {
    const th = document.createElement('th');
    th.textContent = headerText;
    headerRow.appendChild(th);
  });

  tbody.innerHTML = ''; // Clear existing rows
  generalComments.forEach(c => {

    const row = document.createElement('tr');
    row.innerHTML = `
      <td>${c.aidStation || ''}</td>
      <td>${c.messageNum}</td>
      <td>${c.comment}</td>
      <td>${c.operator}</td>
      <td>${c.timestamp}</td>
    `;
    tbody.appendChild(row);
  });
}

// --- Function to convert UTC to custom time zone ---
function convertUTCToCustomTimeZone(utcTimestamp, offset) {
  const date = new Date(utcTimestamp); // Convert UTC timestamp to Date
  const localDate = new Date(date.getTime() + offset * 3600000); // Add offset (in hours)
  return localDate.toLocaleString(); // Format the local date as a string
}

// --- Load general comments and convert timestamps ---
async function loadGeneralComments() {
  try {
    const response = await fetch('load_general_comments.php');
    if (!response.ok) throw new Error('Failed to load data');

    const data = await response.json();

    // Ensure timestamps are converted and formatted after the data is fetched
    generalComments = Array.isArray(data)
      ? data.map(d => {
          // Define the timezone offset (e.g., UTC-8 for Los Angeles)
          const timezoneOffset = -8; // Adjust this for your time zone, e.g., Los Angeles (UTC-8)
          // Convert UTC to local time using the offset
          const formattedTimestamp = convertUTCToCustomTimeZone(d.timestamp, timezoneOffset);

          return {
            aidStation: d.aid_station,
            comment: d.comment,
            operator: d.operator,
            eventName: d.event_name,
            messageNum: d.message_num,
            // Store the converted local timestamp
            timestamp: formattedTimestamp
          };
        })
      : [];
  } catch (e) {
    const saved = JSON.parse(localStorage.getItem('bibData') || '{}');
    generalComments = saved.generalComments || [];
  }

  // Update the table with the converted timestamps
  updateGeneralMessageLog();
}

// --- toggle General Message Log view ---
function toggleGeneralMessageLog() {
  const table = document.getElementById('generalMessageLogTable');
  if (!table) return;
  table.style.display = table.style.display === 'none' ? 'table' : 'none';
  if (table.style.display === 'table') updateGeneralMessageLog();
}

// Function to increment the message number
function incrementMessageNumber() {
  const currentMessageNum = parseInt(localStorage.getItem('msgCounter') || '0', 10);
  const newMessageNum = currentMessageNum + 1;
  localStorage.setItem('msgCounter', newMessageNum);
  return newMessageNum;
}

// Helper to generate CSV content for general messages
function generateCSVContent() {
  const headers = 'Aid Station,Msg #,Comment,Operator,Time';
  const rows = generalComments.map(c => [
    `"${c.aidStation || ''}"`,
    `"${c.messageNum}"`,
    `"${(c.comment || '').replace(/"/g, '""')}"`,  // Escape double quotes
    `"${c.operator || ''}"`,
    `"${c.timestamp || ''}"`
  ].join(','));
  return [headers, ...rows].join('\n');
}

// Helper to generate Winlink content (tab-separated plain text)
function generateWinlinkContent() {
  const headers = 'Aid Station\tMsg #\tComment\tOperator\tTime';
  const rows = generalComments.map(c => [
    `${c.aidStation || ''}`,
    `${c.messageNum}`,
    `${c.comment || ''}`,
    `${c.operator || ''}`,
    `${c.timestamp || ''}`
  ].join('\t'));
  return [headers, ...rows].join('\n');
}

// Export as CSV for general messages
function exportGeneralCSV() {
  const messageNum = incrementMessageNumber();
  const csvContent = generateCSVContent();
  const filename = `${new Date().toISOString().slice(0, 19).replace(/[-T:]/g, '')}_AidStation_${messageNum}.csv`;
  const blob = new Blob([csvContent], { type: 'text/csv' });
  const link = document.createElement('a');
  link.href = URL.createObjectURL(blob);
  link.download = filename;
  link.click();
}

// Export as Winlink (text file) for general messages
function exportGeneralWinlink() {
  const messageNum = incrementMessageNumber();
  const winlinkContent = generateWinlinkContent();
  const filename = `${new Date().toISOString().slice(0, 19).replace(/[-T:]/g, '')}_AidStation_${messageNum}.txt`;
  const blob = new Blob([winlinkContent], { type: 'text/plain' });
  const link = document.createElement('a');
  link.href = URL.createObjectURL(blob);
  link.download = filename;
  link.click();
}

// Export as PDF for general messages (requires jsPDF)
function exportGeneralPDF() {
  const messageNum = incrementMessageNumber();
  const { jsPDF } = window.jspdf;  // Access from the loaded library
  const doc = new jsPDF();
  // Add title
  doc.text('General Message Log', 10, 10);
  // Add content
  let y = 20;
  generalComments.forEach(c => {
    doc.text(`Aid Station: ${c.aidStation || ''}`, 10, y);
    y += 10;
    doc.text(`Msg #: ${c.messageNum}`, 10, y);
    y += 10;
    doc.text(`Comment: ${c.comment || ''}`, 10, y);
    y += 10;
    doc.text(`Operator: ${c.operator || ''}`, 10, y);
    y += 10;
    doc.text(`Time: ${c.timestamp || ''}`, 10, y);
    y += 10;
  });
  // Save
  const filename = `${new Date().toISOString().slice(0, 19).replace(/[-T:]/g, '')}_AidStation_${messageNum}.pdf`;
  doc.save(filename);
}

// --- Auto-refresh Bib + General every 30 seconds ---
setInterval(() => {
  try {
    if (typeof loadData === 'function') {
      loadData();   // updates bib entries and syncs queue
    }
    if (typeof loadGeneralComments === 'function') {
      loadGeneralComments();
    }
    console.log("Auto-refresh fired");
  } catch (e) {
    console.error("Auto-refresh error:", e);
  }
}, 30000);


// === GENERAL MESSAGE EXPORTS (DO NOT TOUCH BIB LOG EXPORT) ===
function exportGeneralCSV() {
  if (generalComments.length === 0) return alert("No general messages.");
  const csv = [
    "Aid Station,Msg #,Comment,Operator,Time",
    ...generalComments.map(c => `"${(c.aidStation||'')}","${c.messageNum}","${(c.comment||'').replace(/"/g,'""')}","${c.operator||''}","${c.timestamp||''}"`)
  ].join("\n");
  downloadFile(csv, `GeneralMsg_${new Date().toISOString().slice(0,10)}.csv`, 'text/csv');
}

function exportGeneralPDF() {
  if (generalComments.length === 0) return alert("No general messages.");
  const { jsPDF } = window.jspdf;
  const doc = new jsPDF();
  doc.setFontSize(16); doc.text("General Message Log", 10, 15);
  doc.setFontSize(10);
  let y = 30;
  generalComments.forEach((c, i) => {
    if (y > 270) { doc.addPage(); y = 20; }
    doc.text(`[${c.messageNum}] ${c.aidStation||'N/A'}: ${c.comment||''}`, 10, y); y += 8;
    doc.text(`    — ${c.operator||''} @ ${c.timestamp||''}`, 15, y); y += 10;
  });
  doc.save(`GeneralMsg_${new Date().toISOString().slice(0,10)}.pdf`);
}

function exportGeneralWinlink() {
  if (generalComments.length === 0) return alert("No general messages.");
  const text = generalComments.map(c => `[Msg ${c.messageNum}] ${c.aidStation||'N/A'}: ${c.comment||''} —${c.operator||''} @${c.timestamp||''}`).join('\n');
  downloadFile(text, `GeneralMsg_Winlink_${new Date().toISOString().slice(0,10)}.txt`, 'text/plain');
}

function downloadFile(content, filename, type) {
  const blob = new Blob([content], { type });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url; a.download = filename; a.click();
  URL.revokeObjectURL(url);
}

/************************************************
 * Export Bib Log CSV using BatchCore
 ************************************************/
function exportBibCSV_v2() {
    if (!window.TVEMC_BatchCore) {
        alert("Message batch core missing.");
        return;
    }

    const batch = TVEMC_BatchCore.createBibBatchForExport("csv");

    if (!batch) {
        alert("No new bib entries to export.");
        return;
    }

    // Convert batch.records → CSV text
    const headerRow = [
        "Bib #", "Action", "Time", "Day", "Station",
        "Comment", "ETA", "Operator", "Date",
        "Event", "First Name", "Last Name",
        "Age", "Gender", "Distance", "Previous Distance"
    ].join(",");

    const csvLines = batch.records.map(e => [
        `"${e.bib_number}"`,
        `"${e.action}"`,
        `"${e.time}"`,
        `"${e.day}"`,
        `"${e.station}"`,
        `"${e.comment}"`,
        `"${e.eta}"`,
        `"${e.operator}"`,
        `"${e.date}"`,
        `"${e.eventName}"`,
        `"${e.first_name}"`,
        `"${e.last_name}"`,
        `"${e.age}"`,
        `"${e.gender}"`,
        `"${e.distance}"`,
        `"${e.previous_distance}"`
    ].join(","));

    const csvContent = [headerRow, ...csvLines].join("\n");

    // Derive filename
    const filename = TVEMC_BatchCore.filenameForBatch(batch, "csv");

    // Trigger download
    const blob = new Blob([csvContent], { type: "text/csv" });
    const url = URL.createObjectURL(blob);

    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);

    URL.revokeObjectURL(url);

    alert(`BIB CSV Exported\nMessage #${batch.message_number}`);
}

/************************************************
 * Export Bib Log as Winlink TXT using BatchCore
 ************************************************/
function exportBibWinlinkTxt_v2() {
    if (!window.TVEMC_BatchCore) {
        alert("Message batch core missing.");
        return;
    }

    const batch = TVEMC_BatchCore.createBibBatchForExport("winlink");

    if (!batch) {
        alert("No new bib entries to export for Winlink.");
        return;
    }

    // Message number padded, e.g. 001
    const msgNumPadded = String(batch.message_number || 0).padStart(3, "0");

    // Build header line for Winlink / RD logs
    const headerLine =
        `MSG:${msgNumPadded} ` +
        `${batch.event_id || "EVENT"} ` +
        `${batch.aid_station_name || "STATION"} ` +
        `OP:${batch.operator || "OPERATOR"} ` +
        `${batch.timestamp_local || ""} ` +
        `BIB EXPORT`;

    // Column header (same fields as CSV)
    const columnHeader = [
        "Bib #", "Action", "Time", "Day", "Station",
        "Comment", "ETA", "Operator", "Date",
        "Event", "First Name", "Last Name",
        "Age", "Gender", "Distance", "Previous Distance"
    ].join(",");

    // Build lines for each record
    const bodyLines = batch.records.map(e => [
        e.bib_number,
        e.action,
        e.time,
        e.day,
        e.station,
        e.comment,
        e.eta,
        e.operator,
        e.date,
        e.eventName,
        e.first_name,
        e.last_name,
        e.age,
        e.gender,
        e.distance,
        e.previous_distance
    ].map(val => String(val == null ? "" : val)).join(","));

    const txtContent = [headerLine, columnHeader, ...bodyLines].join("\n");

    // Derive filename with .txt extension
    const filename = TVEMC_BatchCore.filenameForBatch(batch, "txt");

    // Trigger download as plain text
    const blob = new Blob([txtContent], { type: "text/plain" });
    const url = URL.createObjectURL(blob);

    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);

    URL.revokeObjectURL(url);

    alert(`BIB Winlink TXT Exported\nMessage #${batch.message_number}`);
}
